import { SelectionModel } from '@angular/cdk/collections';
import { Component, OnInit, ViewChild } from '@angular/core';
import { MatPaginator } from '@angular/material/paginator';
import { MatSort } from '@angular/material/sort';
import { MatTableDataSource } from '@angular/material/table';
import { NgbModal, NgbTooltipModule } from '@ng-bootstrap/ng-bootstrap';
import { ToastrService } from 'ngx-toastr';
import { Observable, Subject, debounceTime, distinctUntilChanged, switchMap, takeUntil, tap } from 'rxjs';
import { FormsModule, NgForm } from '@angular/forms';
import Swal from 'sweetalert2';
import { TranslatePipe } from '@ngx-translate/core';
import { Unsub } from '../../../../shared/class/unsub.class';
import { AdminService } from '../../../../shared/services/admin/admin.service';
import { LoaderService } from '../../../../shared/services/loader.service';
import { TranslationService } from '../../../../shared/services/translation.service';

import { NgSelectModule } from '@ng-select/ng-select';
import { MaterialeModules } from '../../../../material.module';
import { RouterLink } from '@angular/router';
import { TranslatePipes } from '../../../../shared/pipes/translate/translate.pipe';
import { AdminPageHeaderComponent } from '../../../../shared/common/pageheader/admin-page-header/admin-page-header.component';

@Component({
  selector: 'app-projects',
  templateUrl: './projects.component.html',
  styleUrls: ['./projects.component.scss'],
  standalone: true,
  imports: [AdminPageHeaderComponent, TranslatePipes, MaterialeModules, NgbTooltipModule, FormsModule, NgSelectModule, RouterLink]
})
export class ProjectsComponent extends Unsub implements OnInit {
  public sendingResponse: { [key: string]: boolean } = {};
  public projects: any;
  public completeprojects: any[] = [];
  public projectid: any;
  public projectname: any;
  private searchTerms = new Subject<string>();
  categories;
  displayedColumns: string[] = ['sno', 'select', 'name', 'actions'];
  dataSource!: MatTableDataSource<any>;
  selection = new SelectionModel<any>(true, []);
  refreshToken$: any;
  permission: any;
  isloader = true;
  public selCat: any = [];
  catCheck;
  @ViewChild(MatPaginator) paginator!: MatPaginator;
  @ViewChild(MatSort) sort!: MatSort;

  constructor(
    private adminService: AdminService,
    private toastr: ToastrService,
    private modalService: NgbModal,
    private translate: TranslatePipe,
    private translationService: TranslationService,
    loaderService: LoaderService
  ) {
    /**
    * This code calls the constructor of the parent class in a subclass. 
    * The super() function is used to initialize the parent class and allows access to its properties and methods. 
    * By calling  super(), the subclass can inherit and use the functionality of the parent class.
    **/
    super(loaderService);
    this.translationService.TranslationStatus.subscribe(() => {
      setTimeout(() => {
        this.paginator._intl.itemsPerPageLabel = this.translate.transform('items per page:');
        this.paginator._intl.nextPageLabel = this.translate.transform('next');
        this.paginator._intl.previousPageLabel = this.translate.transform('previous');
        this.paginator._intl.firstPageLabel = this.translate.transform('first page');
        this.paginator._intl.lastPageLabel = this.translate.transform('last page');
        const of = this.translate.transform('of');
        this.paginator._intl.getRangeLabel = (page: number, pageSize: number, length: number) => {
          if (length == 0 || pageSize == 0) return `0 ${of} ${length}`;
          length = Math.max(length, 0);
          const startIndex = page * pageSize;
          const endIndex = startIndex < length ? Math.min(startIndex + pageSize, length) : startIndex + pageSize;

          return `${startIndex + 1} - ${endIndex} ${of} ${length}`;
        };
      }, 100);
    });
    this.translationService.setTranslationaStatus(true);
  }

  ngOnInit(): void {
    this.allProjects();
    this.adminService.RefreshRquestAdmin$.pipe(
      takeUntil(this.unSubscribe$)
    ).subscribe({
      next: () => {
        this.allProjects();

      }
    });

    this.searchTerms.pipe(
      debounceTime(300),
      distinctUntilChanged(),
      switchMap((): any => this.loadData())
    ).subscribe();
  }
  allProjects() {
    this.isloader = true;
    this.loaderService.showLoader();
    this.adminService
      .projects(1, 10, this.filterValue)
      .pipe(takeUntil(this.unSubscribe$))
      .subscribe({
        next:
          (res) => {
            this.permission = res.permissions;
            this.catCheck = res.check_category;
            this.projects = res.queryList;
            this.categories = res.categories;

            // Assign the data to the data source for the table to render
            this.isloader = false;
            this.dataSource = new MatTableDataSource(this.projects.data);
            if (this.dataSource) {
              this.paginator.pageSize = this.projects.per_page;
              setTimeout(() => {
                this.paginator.pageIndex = this.projects.current_page - 1;
                this.paginator.length = this.projects.total;
              });
              this.dataSource.paginator = this.paginator;
              
              // this.sort.disableClear = true; // Used to disbale the original soting of data if enabled (Only ASC, DES).
              this.dataSource.sort = this.sort;
              this.dataSource.sortingDataAccessor = (data, sortHeaderId) => {
                switch (sortHeaderId) {
                  case 'user' || 'ticketdetails' || 'status':
                    return this.nestedProperty(data, sortHeaderId);
                  default:
                    return this.nestedCaseInsensitive(data, sortHeaderId);
                }
              }
            }
            if (!this.permission?.includes('Project Delete')) {
              this.displayedColumns = ['sno', 'name', 'actions'];
            }
            this.loaderService.hideLoader();
          },
        error: () => {
          this.isloader = true;
          this.loaderService.hideLoader()
        },
      }
      );
  }

  filterValue = "";
  applyFilter(event: Event) {
    this.filterValue = (event.target as HTMLInputElement).value;
    this.searchTerms.next(this.filterValue);
  }

  /** Whether the number of selected elements matches the total number of rows. */
  isAllSelected() {
    const numSelected = this.selection.selected.length;
    let numRows;
    if (this.dataSource) {
        numRows = this.dataSource.data.length;
    }
    return numSelected === numRows;
  }
  /** Selects all rows if they are not all selected; otherwise clear selection. */
  toggleAllRows() {
    if (this.isAllSelected()) {
      this.selection.clear();
      return;
    }

    this.selection.select(...this.dataSource.data);
  }

  /** The label for the checkbox on the passed row */
  checkboxLabel(row?: any): string {
    if (!row) {
      return `${this.isAllSelected() ? 'deselect' : 'select'} all`;
    }
    return `${this.selection.isSelected(row) ? 'deselect' : 'select'} row ${row?.id + 1
      }`;
  }

  addModelPopup(content: any) {
    const modalRef = this.modalService.open(content, { backdrop: 'static', keyboard: false });
    modalRef.result.then(
      () => {
        this.projectid = '';
        this.projectname = '';
      },
      () => {
        this.projectid = '';
        this.projectname = '';
      }
    );
  }
  
  editModelPopup(content: any, row: any) {
    this.projectid = row?.id;
    this.projectname = row?.name;
    const modalRef = this.modalService.open(content, { backdrop: 'static', keyboard: false });
    modalRef.result.then(
      () => {
        this.projectid = '';
        this.projectname = '';
      },
      () => {
        this.projectid = '';
        this.projectname = '';
      }
    );
  }

  addProject(ngForm: NgForm, buttonId) {
    if (ngForm.valid) {
      this.sendingResponse[buttonId] = true;
      const data = {

        id: ngForm.value.id,
        name: ngForm.value.name,
      };
      this.adminService
        .createProjects(data)
        .pipe(takeUntil(this.unSubscribe$))
        .subscribe({
          next:
            (res) => {
              this.sendingResponse[buttonId] = false;
              if (res.errors) {
                this.toastr.error(this.translate.transform(res.errors));
              }
              if (res.success) {
                this.modalService.dismissAll();
                this.toastr.success(this.translate.transform(res.success));
              }
            },
          error: () => {
            this.sendingResponse[buttonId] = false;
          },
        }
        );
    }
  }
  editProject(ngForm: NgForm, buttonId) {
    if (ngForm.valid) {
      this.sendingResponse[buttonId] = true;
      const data = {
        "projects_id": ngForm.value.id,
        "name": ngForm.value.name,
      };
      this.adminService
        .createProjects(data)
        .pipe(takeUntil(this.unSubscribe$))
        .subscribe({
          next:
            (res) => {
              this.sendingResponse[buttonId] = false;
              if (res.errors) {
                this.toastr.error(this.translate.transform(res.errors));
              }
              if (res.success) {
                this.modalService.dismissAll();
                this.toastr.success(this.translate.transform(res.success));
              }
            },
          error: () => {
            this.sendingResponse[buttonId] = false;
          },
        }
        );
    }
  }
  deleteProject(row: any, buttonId) {
    const data = {

      id: row?.id,
    };
    Swal.fire({
      icon: 'warning',
      title: this.translate.transform('Are you sure ?'),
      text: this.translate.transform('Your will not be able to recover!'),
      showCancelButton: true,
      confirmButtonColor: '#6259ca',
      cancelButtonColor: '#f11541',
      confirmButtonText: this.translate.transform('Yes, delete it!'),
      cancelButtonText: this.translate.transform('Close'),
      reverseButtons: true,
    }).then((result: any) => {
      if (result.isConfirmed) {
        this.sendingResponse[buttonId] = true;
        this.adminService
          .deleteProjects(row?.id, data)
          .pipe(takeUntil(this.unSubscribe$))
          .subscribe({
            next:
              (res) => {
                this.sendingResponse[buttonId] = false;
                if (res.success) {
                  this.toastr.success(this.translate.transform(res.success));
                }
              },
            error: () => {
              this.sendingResponse[buttonId] = false;
            },
          }
          );
      }
    });
  }
  
  public assignProjects: any = [];
  assignModelPopup(content: any) {
    this.selCat = [...new Set(this.catCheck)];
    this.adminService
      .allprojects()
      .pipe(takeUntil(this.unSubscribe$))
      .subscribe({
        next: res => {
          res.projects.map((e) => {
            this.assignProjects.push(e.id);
          });
          setTimeout(() => {
            this.completeprojects = res.projects;
          }, 100);

        }
      })
    const modalRef = this.modalService.open(content, { backdrop: 'static', keyboard: false });
    modalRef.result.then(
      () => {
        this.projectid = '';
        this.projectname = '';
        this.assignProjects = [];
        this.selCat = [];
      },
      () => {
        this.projectid = '';
        this.projectname = '';
        this.assignProjects = [];
        this.selCat = [];
      }
    );
  }
  assignProject(ngForm: NgForm, buttonId) {
    if (ngForm.valid) {
      this.sendingResponse[buttonId] = true;
      const data = {
        "projected": this.assignProjects,
        "category_id": ngForm.value.category_id,
      };
      this.adminService
        .projectsassigned(data)
        .pipe(takeUntil(this.unSubscribe$))
        .subscribe({
          next:
            (res) => {
              this.sendingResponse[buttonId] = false;
              if (res.success) {
                this.modalService.dismissAll();
                this.toastr.success(this.translate.transform(res.success));
              }
            },
          error: () => {
            this.sendingResponse[buttonId] = false;
          },
        }
        );
    }
  }
  massDeleteProject(buttonId) {
    if (!this.selection.selected.length) {
      this.toastr.error(this.translate.transform('Please select atleast one Projects'));
      return;
    }
    const id: any = [];
    this.selection.selected.map(function (ele) {
      id.push(ele.id);
    });
    const data = {

      id: id,
    };
    Swal.fire({
      icon: 'warning',
      title: this.translate.transform('Are you sure ?'),
      text: this.translate.transform('Your will not be able to recover!'),
      showCancelButton: true,
      confirmButtonColor: '#6259ca',
      cancelButtonColor: '#f11541',
      confirmButtonText: this.translate.transform('Yes, delete it!'),
      cancelButtonText: this.translate.transform('Close'),
      reverseButtons: true,
    }).then((result: any) => {
      if (result.isConfirmed) {
        this.sendingResponse[buttonId] = true;
        this.adminService
          .massDeleteProjects(data)
          .pipe(takeUntil(this.unSubscribe$))
          .subscribe({
            next: (res) => {
              this.sendingResponse[buttonId] = false;
              this.selection.clear();
              if (res.success) {
                this.toastr.success(this.translate.transform(res.success));
              }
            },
            error: () => {
              this.sendingResponse[buttonId] = false;
            },
          }
          );
      }
    });
  }

  loadData(): Observable<any> {
    this.isloader = true;
    return this.adminService.projects(this.paginator.pageIndex + 1, this.paginator.pageSize, this.filterValue).pipe(
      tap((data: any) => {
        this.isloader = false;
        this.dataSource.data = data.queryList.data;
        setTimeout(() => {
          this.paginator.pageIndex = data.queryList.current_page - 1;
          this.paginator.length = data.queryList.total;
          this.paginator.pageSize = data.queryList.per_page;
        });
      }));
  }

  onPaginateTable() {
    this.selection.clear();
    this.loadData().subscribe();
  }
  override ngOnDestroy() {
    this.modalService?.dismissAll();
  }
  // this is needed to allow sorting on nested properties
  nestedProperty = (data: any, sortHeaderId: string): string | number => {
    return sortHeaderId
      ?.split('.')
      .reduce((accumulator, key) => accumulator && accumulator[key], data) as
      | string
      | number;
  };
  // this is needed to have caseInsensitive sorting
  caseInsensitive = (data: any, sortHeaderId: string): string | number => {
    const value = data[sortHeaderId];
    return typeof value === 'string' ? value.toUpperCase() :
      value;
  };
  nestedCaseInsensitive = (data: any, sortHeaderId: string): string | number => {
    const value = sortHeaderId
      ?.split('.')
      .reduce((accumulator, key) => accumulator && accumulator[key], data) as
      | string
      | number;
    return typeof value === 'string' ? value.toUpperCase() : value;
  };
}
