import { SelectionModel } from '@angular/cdk/collections';
import { Component, ElementRef, OnInit, Renderer2, ViewChild } from '@angular/core';
import { MatPaginator } from '@angular/material/paginator';
import { MatSort } from '@angular/material/sort';
import { MatTableDataSource } from '@angular/material/table';
import { ToastrService } from 'ngx-toastr';
import { Observable, Subject, debounceTime, distinctUntilChanged, switchMap, takeUntil, tap } from 'rxjs';
import { Router, RouterLink } from '@angular/router';
import Swal from 'sweetalert2';
import { DomSanitizer, SafeHtml } from '@angular/platform-browser';
import { TranslatePipe } from '@ngx-translate/core';
import { Unsub } from '../../../shared/class/unsub.class';
import { AdminService } from '../../../shared/services/admin/admin.service';
import { CustomersService } from '../../../shared/services/customers/customers.service';
import { LoaderService } from '../../../shared/services/loader.service';
import { TranslationService } from '../../../shared/services/translation.service';
import { CommonModule } from '@angular/common';
import { NgbAlertModule, NgbTooltipModule } from '@ng-bootstrap/ng-bootstrap';
import { MaterialeModules } from '../../../material.module';
import { TranslatePipes } from '../../../shared/pipes/translate/translate.pipe';
import { DateAgoPipe } from '../../../shared/pipes/date-ago/date-ago.pipe';

@Component({
  selector: 'app-dashboard',
  templateUrl: './dashboard.component.html',
  styleUrls: ['./dashboard.component.scss'],
  standalone : true,
  imports : [CommonModule, TranslatePipes, MaterialeModules, NgbTooltipModule, RouterLink, NgbAlertModule, DateAgoPipe]
})
export class DashboardComponent extends Unsub implements OnInit {
  public customerData: any;
  private searchTerms = new Subject<string>();
  public userId!: string;
  public customer: any;
  public tickets: any;
  public totlaTicketscount: any;
  public activeTicketscount: any;
  public onhoildTicketscount: any;
  public closedTicketscount: any;
  public quotationscount: any;
  public invoicescount: any;
  public refundscount: any;
  mailnotify: any = [];
  displayedColumns: string[] = [
    'id',
    'ticketdetails',
    'status',
    'actions',
  ];
  dataSource!: MatTableDataSource<any>;
  setting: any;
  selection = new SelectionModel<any>(true, []);
  Auth: any;
  isloader = true;
  refreshToken$;
  @ViewChild(MatPaginator) paginator!: MatPaginator;
  @ViewChild(MatSort) sort!: MatSort;
  timezone: any;

  constructor(
    private customersService: CustomersService,
    private adminService: AdminService,
    private router: Router,
    private toastr: ToastrService,
    private translate: TranslatePipe,
    private translationService: TranslationService,
    loaderService: LoaderService,
    private sanitizer: DomSanitizer,
    private renderer: Renderer2,
    private elementRef: ElementRef
  ) {
    /**
    * This code calls the constructor of the parent class in a subclass. 
    * The super() function is used to initialize the parent class and allows access to its properties and methods. 
    * By calling  super(), the subclass can inherit and use the functionality of the parent class.
    **/
    super(loaderService);

    this.dashboard();
    this.adminService.RefreshRquestAdmin$.pipe(
      takeUntil(this.unSubscribe$)
    ).subscribe({
      next: () => {
        this.dashboard();
      },
    });
    this.translationService.TranslationStatus.subscribe(() => {
      setTimeout(() => {
        this.paginator._intl.itemsPerPageLabel = this.translate.transform('items per page:');
        this.paginator._intl.nextPageLabel = this.translate.transform('next');
        this.paginator._intl.previousPageLabel = this.translate.transform('previous');
        this.paginator._intl.firstPageLabel = this.translate.transform('first page');
        this.paginator._intl.lastPageLabel = this.translate.transform('last page');
        const of = this.translate.transform('of');
        this.paginator._intl.getRangeLabel = (page: number, pageSize: number, length: number) => {
          if (length == 0 || pageSize == 0) return `0 ${of} ${length}`;
          length = Math.max(length, 0);
          const startIndex = page * pageSize;
          const endIndex = startIndex < length ? Math.min(startIndex + pageSize, length) : startIndex + pageSize;

          return `${startIndex + 1} - ${endIndex} ${of} ${length}`;
        };
      }, 100);
    });
    this.translationService.setTranslationaStatus(true);
  }

  ngOnInit(): void {

    this.searchTerms.pipe(
      debounceTime(300),
      distinctUntilChanged(),
      switchMap((): any => this.loadData())
    ).subscribe();
  }

  dashboard = () => {
    this.loaderService.showLoader();
    this.isloader = true;
    this.customersService
      .dashboardData(1, 10, this.filterValue)
      .pipe(takeUntil(this.unSubscribe$))
      .subscribe({
        next: (data) => {
          this.Auth = data.authId;
          this.mailnotify = data.mailnotify;

          const keyframes = `
          .content{
            animation-duration : ${this.mailnotify.length * 2}s !important;
          }
            @keyframes scrollAnimation {
              0% {
                transform: translateY(${this.mailnotify.length * 40}%);
              }
              100% {
                transform: translateY(-${(2 * this.mailnotify.length) * 40}%);
              }
            }
          `;
          const styleElement = this.renderer.createElement('style');
          this.renderer.appendChild(styleElement, this.renderer.createText(keyframes));
          this.renderer.appendChild(this.elementRef.nativeElement, styleElement);

          this.tickets = data?.tickets;
          this.setting = data?.setting?.original;
          this.totlaTicketscount = data?.totlaTicketscount;
          this.activeTicketscount = data?.activeTicketscount;
          this.onhoildTicketscount = data?.onhoildTicketscount;
          this.closedTicketscount = data?.closedTicketscount;
          this.quotationscount = data?.quotationscount;
          this.invoicescount = data?.invoicescount;
          this.refundscount = data?.refundscount;
          this.timezone = data.timezonedata;

          // Assign the data to the data source for the table to render
          this.isloader = false;
          this.dataSource = new MatTableDataSource(this.tickets.data);

          if (this.dataSource) {
            this.paginator.pageSize = this.tickets.per_page;
            setTimeout(() => {
              this.paginator.pageIndex = this.tickets.current_page - 1;
              this.paginator.length = this.tickets.total;
            });
            this.dataSource.paginator = this.paginator;
            
              // this.sort.disableClear = true; // Used to disbale the original soting of data if enabled (Only ASC, DES).
              this.dataSource.sort = this.sort;
              this.dataSource.sortingDataAccessor = (data, sortHeaderId) => {
                switch (sortHeaderId) {
                  case 'user':
                  case 'ticketdetails':
                  case 'status':
                    return this.nestedProperty(data, sortHeaderId);
                  default:
                    return this.nestedCaseInsensitive(data, sortHeaderId);
                }
              }
          }
          this.loaderService.hideLoader();
        },
        error: () => { this.isloader = true; this.loaderService.hideLoader() }
      });
  };
  filterValue = "";
  applyFilter(event: Event) {
    this.filterValue = (event.target as HTMLInputElement).value;
    this.searchTerms.next(this.filterValue);
  }

  showTicketDetails(id: any) {
    this.customersService
      .showTicket(id)
      .pipe(takeUntil(this.unSubscribe$))
      .subscribe({
        next: (event) => {
          this.router.navigate([
            `/customer/ticket-view/${event.ticket.enc_ticket_id}`,
          ]);
        },
      });
  }

  sendMarkRequest(id) {
    const data = {
      id: id,
    };
    this.customersService.markNotification(data)
      .pipe(takeUntil(this.unSubscribe$))
      .subscribe({ next: () => { } });
  }

  loadData(): Observable<any> {
    this.isloader = true;
    return this.customersService
      .dashboardData(this.paginator.pageIndex + 1, this.paginator.pageSize, this.filterValue).pipe(
        tap((data: any) => {
          this.isloader = false;
          this.dataSource.data = data.tickets.data;
          setTimeout(() => {
            this.paginator.pageIndex = data.tickets.current_page - 1;
            this.paginator.length = data.tickets.total;
            this.paginator.pageSize = data.tickets.per_page;
          });
        }));
  }

  onPaginateTable() {
    this.selection.clear();
    this.loadData().subscribe();
  }
  sanitizeHtml(html: string): SafeHtml {
    return this.sanitizer.bypassSecurityTrustHtml(html);
  }
  
  animationPlayState: string = 'running';

  stopScroll() {
    this.animationPlayState = 'paused';
  }

  startScroll() {
    this.animationPlayState = 'running';
  }
  // this is needed to allow sorting on nested properties
  nestedProperty = (data: any, sortHeaderId: string): string | number => {
    return sortHeaderId
      ?.split('.')
      .reduce((accumulator, key) => accumulator && accumulator[key], data) as
      | string
      | number;
  };
  // this is needed to have caseInsensitive sorting
  caseInsensitive = (data: any, sortHeaderId: string): string | number => {
    const value = data[sortHeaderId];
    return typeof value === 'string' ? value.toUpperCase() :
      value;
  };
  nestedCaseInsensitive = (data: any, sortHeaderId: string): string | number => {
    const value = sortHeaderId
      ?.split('.')
      .reduce((accumulator, key) => accumulator && accumulator[key], data) as
      | string
      | number;
    return typeof value === 'string' ? value.toUpperCase() : value;
  };
}
