<?php

namespace App\Http\Controllers\Admin;

use Carbon\Carbon;
use Stripe\Stripe;
use App\Models\User;
use App\Jobs\MailSend;
use App\Models\Setting;
use App\Models\Apptitle;
use App\Models\Customer;
use Stripe\PaymentIntent;
use App\Models\Footertext;
use App\Models\Seosetting;
use App\Models\AllInvoices;
use Jenssegers\Agent\Agent;
use Illuminate\Http\Request;
use App\Models\RefundComment;
use App\Models\Ticket\Ticket;
use App\Models\AllTransactions;
use App\Models\Ticket\Category;
use Torann\GeoIP\Facades\GeoIP;
use App\Models\EmployeeActivity;
use App\Models\CustomInvoiceFields;
use App\Http\Controllers\Controller;
use App\Models\Currency;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Srmklive\PayPal\Services\PayPal;
use Illuminate\Support\Facades\Crypt;
use App\Notifications\PaidTicketNotifications;
use Illuminate\Support\Facades\Schema;
use Spatie\Permission\Models\Role;

class AdminPaymentsController extends Controller
{

    public function adminInvoiceDetails($id)
    {
        $this->authorize('Custom Invoices Access');

        $transaction = AllTransactions::where('id', decrypt($id))->first();
        $data['transaction'] = $transaction;


        return $data;
    }

    public function customInvoicesStore(Request $request)
    {

        $this->authorize(ability: 'Custom Invoices Access');

        if ((setting('enablePaypal') == 'on' || setting('enableStripe') == 'on')) {

            $sets = [];

            // Loop through the input data to group sets of title, description, and price
            foreach ($request->all() as $key => $value) {
                // Extract the key without the number
                $keyWithoutNumber = preg_replace('/[0-9]+/', '', $key);
                if (in_array($keyWithoutNumber, ['title', 'price', 'quantity'])) {
                    // Extract the index from the key
                    $index = preg_replace('/[^0-9]/', '', $key);

                    // Initialize an array for the set if it doesn't exist yet
                    if (!isset($sets[$index])) {
                        $sets[$index] = [];
                    }

                    // Add the value to the set array
                    $sets[$index][$keyWithoutNumber] = $value;
                }
            }
            $totalPrice = 0;
            $subtotal = 0;
            foreach ($sets as $set) {
                // Extract the price and quantity for each row
                $price = isset($set['price']) ? floatval($set['price']) : 0;
                $quantity = isset($set['quantity']) ? intval($set['quantity']) : 0;
                $amount = $price * $quantity;
                $totalPrice += $amount;
            }
            $taxPercent = $request->input('taxPercent', 0);
            $subtotal = number_format($totalPrice, 2);
            if ($taxPercent) {
                $totalPrice += ($totalPrice * floatval($taxPercent)) / 100;
            }
            $finalPrice = number_format($totalPrice, 2);

            $lastTransactionId = AllTransactions::latest()->value('invoiceId') != null ? AllTransactions::latest()->value('invoiceId') : 1;
            $customIdInDb = (int)Setting::where('key', 'customInvoiceId')->value('value');

            if ($lastTransactionId >= $customIdInDb) {
                $invoiceId = $lastTransactionId + 1; // increment
            } else {
                $invoiceId = $customIdInDb; // increment
            }

            $transaction = new AllTransactions();
            $transaction->cust_id = $request->customer;
            $transaction->ticket_type = 'customInvoice';
            $transaction->invoiceId = $invoiceId;
            $transaction->dateIssued = Carbon::now()->format('Y-m-d');
            $paidDate = $request->input('paidDate');
            $transaction->payment_date = ($paidDate !== null && $paidDate !== 'undefined') ? $paidDate : null;

            // $transaction->payment_date = ($request->input('paidDate') != null && $request->input('paidDate') != undefined) ? $request->input('paidDate') : null;
            $transaction->status = $request->status != null ? $request->status : 'pending';
            $transaction->ticket_id = $request->input('ticket');
            $transaction->amount = $finalPrice;
            if ($request->status == 'paid') {
                $transaction->currency = 'USD';
                $symbol = config('currencies.' . $transaction->currency);
                $transaction->customerAmount = $symbol . ' ' . $finalPrice;
                $transaction->payment_method = 'Offline Payment';
            }

            $transaction->save();
            $logo = null;
            if ($request->hasFile('invoiceLogo')) {
                $file = $request->file('invoiceLogo');
                if ($file->isValid()) {
                    $destinationPath = public_path('/uploads/invoice');
                    if (!File::exists($destinationPath)) {
                        File::makeDirectory($destinationPath, 0777, true);
                    }
                    $logo = date('Ymdhis') . "." . $file->getClientOriginalExtension();
                    $file->move($destinationPath, $logo);
                } else {
                    return response()->json(['error' => 'The uploaded file is invalid or empty.']);
                }
            }

            $allInvoices = new AllInvoices();
            $allInvoices->parentId = $transaction->id;
            $allInvoices->companyName = $request->input('companyName');
            $allInvoices->companyLogo = $logo;
            $allInvoices->customInvoiceId = $invoiceId;
            $allInvoices->addressLine1 = $request->input('addressLine1');
            $allInvoices->addressLine2 = $request->input('addressLine2');
            $allInvoices->cityName = $request->input('cityName');
            $allInvoices->stateName = $request->input('stateName');
            $allInvoices->zipPostalCode = $request->input('zipPostalCode');
            $allInvoices->countryName = $request->input('countryName');
            $allInvoices->taxName = $request->input('taxName');
            $allInvoices->taxPercent = $request->input('taxPercent');
            $allInvoices->taxId = $request->input('taxId');
            $allInvoices->customerName = $request->input('customer');
            $allInvoices->ticket_id = $request->input('ticket');
            $allInvoices->clcompanyName = $request->input('clcompanyName');
            $allInvoices->status = $request->input('status');
            $allInvoices->paidDate = $request->input('paidDate');
            $allInvoices->clemail = $request->input('clemail');
            $allInvoices->clmobile = $request->input('clmobile');
            $allInvoices->cltaxName = $request->input('cltaxName');
            $allInvoices->cltaxId = $request->input('cltaxId');
            $allInvoices->claddress = $request->input('claddress');
            $allInvoices->clcountry = $request->input('clcountry');
            $allInvoices->invoicenote = $request->input('invoicenote');
            $allInvoices->invoicetc = $request->input('invoicetc');
            $allInvoices->subtotal = $subtotal;
            $allInvoices->totalFinalPrice = $finalPrice;


            $allInvoices->save();

            $geolocation = GeoIP::getLocation(request()->getClientIp());
            $agent = new Agent();
            $activity = new EmployeeActivity();
            $activity->user_id = Auth::user()->id;
            $activity->activity_type = 'Created a new custom invoice';
            $activity->ip_address = $geolocation->ip;
            $activity->browser = $agent->browser();
            $activity->device = $agent->device();
            $activity->save();

            foreach ($sets as $set) {
                // Check if all necessary fields are present
                if (isset($set['title'], $set['price'], $set['quantity'])) {
                    $customInvoiceFields = new CustomInvoiceFields();
                    $customInvoiceFields->title = $set['title'];
                    $customInvoiceFields->price = $set['price'];
                    $customInvoiceFields->quantity = $set['quantity'];
                    $customInvoiceFields->amount = $set['price'] * $set['quantity'];
                    $customInvoiceFields->parentId = $transaction->id;
                    $customInvoiceFields->save();
                }
            }

            $dueDate = '';
            $customer = Customer::find($request->customer);
            if ($transaction->status == 'pending' && setting('autoInvoiceCancelEnable') == 'on') {
                $autocancelDays = setting('invoiceCancelDays');
                $dueDate = date(setting('date_format'), strtotime("+$autocancelDays days", strtotime($transaction->dateIssued)));
            }

            $ticketData = [
                'username' => $customer->username,
                'invoiceId' => $transaction->invoiceId,
                'invoice_url' => url('/customer/custom-invoice/' . encrypt($transaction->id)),
                'dueDate' => $dueDate ? $dueDate : '',
            ];
            dispatch((new MailSend($customer->email, 'when_a_customInvoice_is_created', $ticketData)));
            $transaction->customer->notify(new PaidTicketNotifications($transaction));


            $superAdminCount = User::with('roles')->get()->filter(
                fn($user) => $user->roles->where('name', 'superadmin')->toArray()
            );

            foreach ($superAdminCount as $admin) {
                $admin->notify(new PaidTicketNotifications($transaction));
            }

            return response()->json(["redirect" => "customInvoices", 'success' => 'Custom invoice created successfully.']);
        } else {
            return response()->json(["redirect" => "customInvoices", 'error' => 'Please enable any payment gateway before creating an invoice.']);
        }
    }

    public function getDetails($id)
    {

        $customer = Customer::where('id', $id)->where('status', '1')->where('verified', '1')->first();
        $tickets = Ticket::where('cust_id', $id)
            ->whereIn('status', ['New', 'Inprogress'])
            ->doesntHave('invoice')
            ->get();
        if ($customer) {
            return response()->json([
                'success' => true,
                'data' => [
                    'firstname' => $customer->firstname,
                    'lastname' => $customer->lastname,
                    'companyName' => $customer->companyName,
                    'taxName' => $customer->taxName,
                    'taxId' => $customer->taxId,
                    'Address' => $customer->Address,
                    'email' => $customer->email,
                    'phone' => $customer->phone,
                    'country' => $customer->country,
                ],
                'tickets' => $tickets,
            ]);
        }
        return response()->json(['success' => false, 'message' => 'Customer not found'], 404);
    }

    public function deleteInvoice($id)
    {
        $this->authorize('Invoice Delete Access');
        $invoice = AllTransactions::findOrFail($id);
        $invoice->delete();

        $geolocation = GeoIP::getLocation(request()->getClientIp());
        $agent = new Agent();
        $activity = new EmployeeActivity();
        $activity->user_id = Auth::user()->id;
        $activity->activity_type = 'Deleted custom invoice' . $id;
        $activity->ip_address = $geolocation->ip;
        $activity->browser = $agent->browser();
        $activity->device = $agent->device();
        $activity->save();


        return response()->json(['success' => 'custom invoice deleted successfully']);
    }

    public function cancelInvoice($id)
    {

        $this->authorize('Invoice Cancel Access');
        $invoice = AllTransactions::findOrFail($id);
        $invoice->status = 'cancelled';
        $invoice->save();

        $geolocation = GeoIP::getLocation(request()->getClientIp());
        $agent = new Agent();
        $activity = new EmployeeActivity();
        $activity->user_id = Auth::user()->id;
        $activity->activity_type = 'Cancelled custom invoice' . $id;
        $activity->ip_address = $geolocation->ip;
        $activity->browser = $agent->browser();
        $activity->device = $agent->device();
        $activity->save();

        return response()->json(['success' => 'custom invoice cancelled successfully', 'invoice' => $invoice]);
    }

    public function refundSettings(Request $request)
    {
        $this->authorize('Payment Gateways Access');

        if ($request->has('enableRefund')) {
            $this->validate($request, [
                'refundDays' => 'required|gte:1',
            ]);
        }

        $data['enableRefund'] = $request->enableRefund ? 'on' : 'off';
        $data['refundDays'] = $request->refundDays;

        $this->updateSettings($data);

        $geolocation = GeoIP::getLocation(request()->getClientIp());
        $agent = new Agent();
        $activity = new EmployeeActivity();
        $activity->user_id = Auth::user()->id;
        $activity->activity_type = 'Updated refund Settings.';
        $activity->ip_address = $geolocation->ip;
        $activity->browser = $agent->browser();
        $activity->device = $agent->device();
        $activity->save();

        return response()->json(['success' => 'Updated successfully']);
    }

    public function autoInvoiceCancelSetting(Request $request)
    {

        $this->authorize('Invoice Cancel Access');
        if ($request->has('autoInvoiceCancelEnable')) {
            $this->validate($request, [
                'invoiceCancelDays' => 'required|gte:1',
            ]);
        }

        $data['autoInvoiceCancelEnable'] = $request->autoInvoiceCancelEnable ? 'on' : 'off';
        $data['invoiceCancelDays'] = $request->invoiceCancelDays;

        $this->updateSettings($data);

        $geolocation = GeoIP::getLocation(request()->getClientIp());
        $agent = new Agent();
        $activity = new EmployeeActivity();
        $activity->user_id = Auth::user()->id;
        $activity->activity_type = 'Updated Auto Invoice Cancel Settings.';
        $activity->ip_address = $geolocation->ip;
        $activity->browser = $agent->browser();
        $activity->device = $agent->device();
        $activity->save();

        return response()->json(['success' => 'Updated successfully']);
    }

    public function autoInvoicedeleteSetting(Request $request)
    {

        $this->authorize('Invoice Delete Access');
        if ($request->has('autoinvoicedelete')) {
            $this->validate($request, [
                'autodeleteinvoiceDays' => 'required|gte:1',
            ]);
        }

        $data['autoinvoicedelete'] = $request->autoinvoicedelete ? 'on' : 'off';
        $data['autodeleteinvoiceDays'] = $request->autodeleteinvoiceDays;

        $this->updateSettings($data);

        $geolocation = GeoIP::getLocation(request()->getClientIp());
        $agent = new Agent();
        $activity = new EmployeeActivity();
        $activity->user_id = Auth::user()->id;
        $activity->activity_type = 'Updated Auto Delete invoice Settings.';
        $activity->ip_address = $geolocation->ip;
        $activity->browser = $agent->browser();
        $activity->device = $agent->device();
        $activity->save();

        return response()->json(['success' => 'Updated successfully']);
    }

    private function updateSettings($data)
    {

        foreach ($data as $key => $val) {
            $setting = Setting::where('key', $key);
            if ($setting->exists())
                $setting->first()->update(['value' => $val]);
        }
    }

    public function refundRequests(Request $request)
    {
        $this->authorize(('Payments Access'));
        $this->authorize('Refund Access');

        // $transaction = AllTransactions::whereIn('status', ['refunded', 'refundpending', 'refundRejected'])->with('customer')->latest('updated_at')->get();
        // $transaction = $transaction->each(function ($trans) {
        //     $trans->encrypted_id = Crypt::encrypt($trans->id);
        //     return $trans;
        // });
        // $data['transaction'] = $transaction;

        $customInvoices = AllTransactions::whereIn('status', ['refunded', 'refundpending', 'refundRejected'])
            ->with(['customInvoiceElements', 'customer'])
            ->latest('updated_at');

        $searchTerm = $request['filter'] ?? null;
        $perPage    = $request['per_page'] ?? 15;
        $page       = $request['page'] ?? 1;

        if ($searchTerm) {
            $customInvoices->where(function ($query) use ($searchTerm) {
                $columns = Schema::getColumnListing('all_transactions');
                foreach ($columns as $column) {
                    $query->orWhere($column, 'LIKE', "%{$searchTerm}%");
                }
            });
        }

        $items = $customInvoices->paginate($perPage);

        $items->getCollection()->each(fn($trans) => $trans->encrypted_id = Crypt::encrypt($trans->id));

        $data['queryList'] = $items;

        $data['setting'] = ['date_format' => setting('date_format'), 'time_format' => setting('time_format'), 'default_timezone' => setting('default_timezone')];
        $data['timezonedata'] = Auth::user()->timezone ?? setting('default_timezone');

        $currency = Currency::first();
        $configData = config('currencies')[$currency->code];
        $data['configData'] = $configData;


        $user = User::with('usetting')->findOrFail(Auth::user()->id);
        $role = Role::where('name', $user->getRoleNames()[0])->get();
        foreach ($role as $roles) {
            $data['permissions'] = $roles->getPermissionNames();
        }
        return $data;
    }

    public function refundRequestsView($id)
    {

        $this->authorize('Refund Access');


        $refundItem = AllTransactions::with('customer', 'refundComments')->where('id', decrypt($id))->whereIn('status', ['refunded', 'refundpending', 'refundRejected'])->first();
        $data['refundItem'] = $refundItem;

        $currency = $refundItem->currency;
        $data['symbol'] = config('currencies.' . $currency);

        $data['setting'] = ['date_format' => setting('date_format'),'time_format' => setting('time_format'), 'default_timezone' => setting('default_timezone')];
        $data['timezonedata'] = Auth::user()->timezone ?? setting('default_timezone');

        $data['user'] = Auth::user();

        return $data;
    }

    public function refundPostComment(Request $request)
    {

        try {
            $this->authorize('Refund Access');
            $refundItem = AllTransactions::where('id', decrypt($request->data['id']))->first();
            if ($refundItem->status == 'refundpending') {

                $com = $request->data['commentReply'];
                $co = preg_replace('/^(<p><br><\\/p>)+|(<p><br><\\/p>)+$/', '', $com);
                $co = preg_replace('/(<div><br><\/div>)+/', '<br>', $co);
                $co = trim($co);
                $refundComent = RefundComment::create([
                    'user_id' => Auth::user()->id,
                    'refund_id' => decrypt($request->data['id']),
                    'comment' => $co,
                ]);
                // $refundComent->customer->notify(new PaidTicketNotifications($refundItem));

                $geolocation = GeoIP::getLocation(request()->getClientIp());
                $agent = new Agent();
                $activity = new EmployeeActivity();
                $activity->user_id = Auth::user()->id;
                $activity->activity_type = 'Added a comment in refund request #' . $refundItem->invoiceId;
                $activity->ip_address = $geolocation->ip;
                $activity->browser = $agent->browser();
                $activity->device = $agent->device();
                $activity->save();

                // event(new \App\Events\ConversationEvent($refundItem->customer, url('/customer/refundView/' . encrypt($refundItem->id)), $refundItem->customer->id, null, 'New Comment for your refund request on invoice #' . $refundItem->invoiceId, 'custNotification', null));

                $ticketData = [
                    'username' => $refundItem->customer->username,
                    'invoice_id' => $refundItem->invoiceId,
                    'refund_request_page_url' => url('/customer/refund-view/' . encrypt($refundItem->id))
                ];
                dispatch((new MailSend($refundItem->customer->email, 'customer_send_refund_comment_from_admin', $ticketData)));

                return response()->json(['success' => true]);
            }
            return response()->json(['error' => true]);
        } catch (\Throwable $th) {
            return response()->json(['error' => $th->getMessage()]);
        }
    }

    public function rejectRefund(Request $request)
    {
        try {
            $this->authorize('Refund Access');
            $refundItem = AllTransactions::where('id', decrypt($request->data['id']))->where('status', 'refundpending')->first();

            if (isset($refundItem)) {
                $refundItem->status = 'refundRejected';
                $refundItem->update();

                $geolocation = GeoIP::getLocation(request()->getClientIp());
                $agent = new Agent();
                $activity = new EmployeeActivity();
                $activity->user_id = Auth::user()->id;
                $activity->activity_type = 'Rejected a refund for' . ' ' . $refundItem->invoiceId;
                $activity->ip_address = $geolocation->ip;
                $activity->browser = $agent->browser();
                $activity->device = $agent->device();
                $activity->save();

                $refundItem->customer->notify(new PaidTicketNotifications($refundItem));

                // event(new \App\Events\ConversationEvent($refundItem->customer, url('/customer/refundView/' . encrypt($refundItem->id)), $refundItem->customer->id, null, 'Your refund request is rejected for invoice id #' . $refundItem->invoiceId, 'custNotification', null));

                $ticketData = [
                    'username' => $refundItem->customer->username,
                    'invoice_id' => $refundItem->invoiceId,
                ];

                dispatch((new MailSend($refundItem->customer->email, 'customer_refund_rejected', $ticketData)));

                return response()->json(['success' => lang('successfully updated')]);
            }
            return response()->json(['itemNotFound' => true]);
        } catch (\Throwable $th) {
            return response()->json(['error' => $th->getMessage()]);
        }
    }

    public function approveRefund(Request $request)
    {
        try {
            $this->authorize('Refund Access');
            $refundItem = AllTransactions::where('id', decrypt($request->data['id']))->where('status', 'refundpending')->first();

            if (isset($refundItem)) {

                if ($refundItem->payment_method == 'Offline Payment') {
                    $refundItem->status = 'refunded';
                    $refundItem->netAmountRefunded = $refundItem->amount;
                    $refundItem->save();

                    $geolocation = GeoIP::getLocation(request()->getClientIp());
                    $agent = new Agent();
                    $activity = new EmployeeActivity();
                    $activity->user_id = Auth::user()->id;
                    $activity->activity_type = 'Approved refund for #' . $refundItem->invoiceId;
                    $activity->ip_address = $geolocation->ip;
                    $activity->browser = $agent->browser();
                    $activity->device = $agent->device();
                    $activity->save();

                    $refundItem->customer->notify(new PaidTicketNotifications($refundItem));
                    $ticketData = [
                        'username' => $refundItem->customer->username,
                        'invoice_id' => $refundItem->invoiceId,
                    ];
                    dispatch((new MailSend($refundItem->customer->email, 'customer_refund_approved', $ticketData)));

                    return response()->json(['success' => 'Refund issued successfully.']);
                }


                if ($refundItem->payment_method == 'Stripe') {
                    Stripe::setApiKey(setting('stripeSecret'));
                    $stripe = new \Stripe\StripeClient(setting('stripeSecret'));

                    $paymentIntent = PaymentIntent::retrieve($refundItem->transaction_id);
                    $chargeId = $paymentIntent->latest_charge;
                    $ch = $stripe->charges->retrieve(
                        $chargeId,
                        [],
                        ['api_key' => setting('stripeSecret')]
                    );
                    $balanceTransactionId = $ch->balance_transaction;
                    $balanceTransaction = \Stripe\BalanceTransaction::retrieve($balanceTransactionId);
                    $exchangeRate = $balanceTransaction->exchange_rate ? $balanceTransaction->exchange_rate : 1;

                    $refundedamount = number_format($balanceTransaction->net / 100, 2);
                    $refundedamount = (float) str_replace(',', '', $refundedamount);
                    $finalRefundAmount = number_format($refundedamount / $exchangeRate, 2);
                    $finalRefundAmount = str_replace(',', '', $finalRefundAmount);
                    $gatewayFee = number_format($balanceTransaction->fee_details[0]->amount / 100, 2);
                    $gatewayFee = (float) str_replace(',', '', $gatewayFee);
                    $finalGatewayFee = number_format($gatewayFee / $exchangeRate, 2);
                    $taxFee = number_format($balanceTransaction->fee_details[1]->amount / 100, 2);
                    $taxFee = (float) str_replace(',', '', $taxFee);
                    $finalTaxFee = number_format($taxFee / $exchangeRate, 2);
                    $refundAmt = (float) $finalRefundAmount;

                    try {
                        $refundRequest = \Stripe\Refund::create([
                            'charge' => $chargeId,
                            'amount' => $refundAmt * 100,
                        ]);

                        $refundItem->refundTxId = $refundRequest->id;
                        $refundItem->gatewayName = $balanceTransaction->fee_details[0]->description;
                        $refundItem->gatewayFee = (float) str_replace(',', '', $finalGatewayFee);
                        $refundItem->countryTaxName = $balanceTransaction->fee_details[1]->description;
                        $refundItem->countryTaxAmount = (float) str_replace(',', '', $finalTaxFee);
                        $refundItem->netAmountRefunded = (float) str_replace(',', '', $finalRefundAmount);
                        $refundItem->status = 'refunded';
                        $refundItem->save();
                    } catch (\Exception $e) {
                        return response()->json(['error' => $e->getMessage()]);
                    }
                } else {
                    $configPaypal = Setting::paypal();
                    $provider = new PayPal($configPaypal);

                    $provider->getAccessToken();

                    $provider->setCurrency($refundItem->currency);

                    $response = $provider->showCapturedPaymentDetails($refundItem->transaction_id);

                    $paypalFee = $response['seller_receivable_breakdown']['paypal_fee']['value'];
                    $netAmount = $response['seller_receivable_breakdown']['net_amount']['value'];


                    if ($response['status'] == "COMPLETED") {

                        $refundItem->refundTxId = $response['id'];
                        $refundItem->gatewayName = lang('Paypal Processing Fee');
                        $refundItem->gatewayFee = (float) $paypalFee;
                        $refundItem->netAmountRefunded = round((float) $netAmount, 2);
                        $refundItem->status = 'refunded';
                        $refundItem->save();
                    }


                    $response = $provider->refundCapturedPayment(
                        $refundItem->transaction_id,
                        $refundItem->invoiceId,
                        $netAmount,
                        $refundItem->refundReason
                    );
                }


                $geolocation = GeoIP::getLocation(request()->getClientIp());
                $agent = new Agent();
                $activity = new EmployeeActivity();
                $activity->user_id = Auth::user()->id;
                $activity->activity_type = 'Approved refund for #' . $refundItem->invoiceId;
                $activity->ip_address = $geolocation->ip;
                $activity->browser = $agent->browser();
                $activity->device = $agent->device();
                $activity->save();

                $refundItem->customer->notify(new PaidTicketNotifications($refundItem));

                // event(new \App\Events\ConversationEvent($refundItem->customer, url('/customer/refundView/' . encrypt($refundItem->id)), $refundItem->customer->id, null, 'Your refund request is approved for invoice id #' . $refundItem->invoiceId, 'custNotification', null));

                $ticketData = [
                    'username' => $refundItem->customer->username,
                    'invoice_id' => $refundItem->invoiceId,
                ];
                dispatch((new MailSend($refundItem->customer->email, 'customer_refund_approved', $ticketData)));

                return response()->json(['success' => 'Refund issued successfully.']);
            }
            return response()->json(['itemNotFound' => true]);
        } catch (\Throwable $th) {
            return response()->json(['error' => $th->getMessage()]);
        }
    }
}
