<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

use App\Models\Seosetting;
use App\Models\Apptitle;
use App\Models\Footertext;
use App\Models\Languages;
use App\Models\Translate;
use Validator;
use App\Models\Setting;
use App\Models\User;
use Auth;
use Illuminate\Support\Facades\Schema;
use Spatie\Permission\Models\Role;
use Jenssegers\Agent\Agent;
use Torann\GeoIP\Facades\GeoIP;
use App\Models\EmployeeActivity;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Response;
use Stichoza\GoogleTranslate\GoogleTranslate;

class LanguagesController extends Controller
{
    public function index(Request $request)
    {
        $this->authorize('Languages Access');

        $languages = Languages::withCount(['translates' => function ($query) {
            $query->where('value', null);
        }]);

        $searchTerm = $request['filter'] ?? null;
        $perPage    = $request['per_page'] ?? 15;
        $page       = $request['page'] ?? 1;

        if ($searchTerm) {
            $languages->where(function ($query) use ($searchTerm) {
                $columns = Schema::getColumnListing('languages');
                foreach ($columns as $column) {
                    $query->orWhere($column, 'LIKE', "%{$searchTerm}%");
                }
            });
        }
        $items = $languages->paginate($perPage);
        $data['queryList'] = $items;
        $data['setting'] = response([
            'default_lang' => setting('default_lang'),
            'date_format' => setting('date_format'),
        ]);
        $user = User::with('usetting')->findOrFail(Auth::user()->id);
        $role = Role::where('name', $user->getRoleNames()[0])->get();
        foreach ($role as $roles) {
            $data['permissions'] = $roles->getPermissionNames();
        }


        return $data;
    }
    public function defaultlang()
    {
        if (Auth::user()) {
            $user = User::with('usetting')->findOrFail(Auth::user()->id);
            $role = Role::where('name', $user->getRoleNames()[0])->get();
            foreach ($role as $roles) {
                $data['permissions'] = $roles->getPermissionNames();
            }
        }
        $languages = Languages::get();
        $data['languages'] = $languages;
        $data['default_lang'] = setting('default_lang');
        return $data;
    }

    public function create()
    {
        $this->authorize('Languages Create');

        $title = Apptitle::first();
        $data['title'] = $title;

        $footertext = Footertext::first();
        $data['footertext'] = $footertext;

        $seopage = Seosetting::first();
        $data['seopage'] = $seopage;

        return $data;
    }

    public function edit($lang)
    {

        $this->authorize('Languages Edit');


        $language = Languages::findOrFail($lang);
        $data['language'] = $language;

        $data['default_lang'] = setting('default_lang');

        return $data;
    }

    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'languagename' => ['required', 'string', 'max:150'],
            'languagenativename' => ['required', 'string', 'max:150'],
        ]);

        if ($validator->fails()) {
            return $validator->errors()->all();
        }

        $language = Languages::findOrFail($id);
        if (!$request->has('is_default')) {
            if ($language->languagecode == setting('default_lang')) {
                return response()->json(['error' => 'Is Default Language']);
            }
        }
        $update = Languages::where('id', $id)->update([
            'languagename' => $request->languagename,
            'languagenativename' => $request->languagenativename,
            'is_rtl' => $request->is_rtl ?  '1' : '0',
        ]);
        if ($update) {
            if ($request->has('is_default')) {
                $data['default_lang'] = removeSpaces($language->languagecode);
                $this->updateSettings($data);
            }

            $geolocation = GeoIP::getLocation(request()->getClientIp());
            $agent = new Agent();
            $activity = new EmployeeActivity();
            $activity->user_id = Auth::user()->id;
            $activity->activity_type = 'Updated languages';
            $activity->ip_address = $geolocation->ip;
            $activity->browser = $agent->browser();
            $activity->device = $agent->device();
            $activity->save();
            return response()->json(['success' => 'Updated Successfully']);
        }
    }

    /**
     * insert the value in translate.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Language  $language
     * @return \Illuminate\Http\Response
     */
    public function add(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'key' => ['required', 'string', 'max:150'],
            'value' => ['required', 'string', 'max:150'],
        ]);

        if ($validator->fails()) {
            return $validator->errors()->all();
        }


        $languages = Languages::all();
        if (Translate::where('lang_code', $id)->where('key', $request->key)->first()) {
            return response(['error' => 'Key is already avaliable']);
        }



        Translate::create([
            'lang_code' => $id,
            'group_langname' => $request->group_langname,
            'key' => $request->key,
            'value' => $request->value,
        ]);

        foreach ($languages as $language) {
            if ($language->languagecode != $id) {
                Translate::create([
                    'lang_code' => $language->languagecode,
                    'group_langname' => $request->group_langname,
                    'key' => $request->key,
                    'value' => null,
                ]);
            }
        }
        return response()->json(['success' => 'Insert Successfully']);
    }
    public function addOnNotFound(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'key' => ['required', 'string', 'max:150'],
        ]);

        if ($validator->fails()) {
            return $validator->errors()->all();
        }


        $languages = Languages::all();
        if (Translate::where('lang_code', $id)->where('key', $request->key)->first()) {
            return response(['error' => 'Key is already avaliable']);
        }


        foreach ($languages as $language) {
            Translate::create([
                'lang_code' => $language->languagecode,
                'group_langname' => $request->group_langname,
                'key' => $request->key,
                'value' => null,
            ]);
        }
        return response()->json(200);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'languagename' => ['required', 'string', 'max:150'],
            'languagenativename' => ['required', 'string', 'max:150'],
            'languagecode' => ['required', 'string', 'max:10', 'min:2', 'unique:languages'],
        ]);
        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()]);
        }
        if (!array_key_exists($request->languagecode, languages())) {

            return response()->json(['error' => 'Something went wrong please try again']);
        }

        $create = Languages::create([
            'languagename' => $request->languagename,
            'languagenativename' => $request->languagenativename,
            'languagecode' => $request->languagecode,
            'is_rtl' => $request->is_rtl ?  '1' : '0',
        ]);



        if ($create) {
            $translates = Translate::where('lang_code', setting('default_lang'))->get();

            foreach ($translates as $translate) {
                $value = ($request->languagecode == "en") ? $translate->key : null;
                $lang = new Translate();
                $lang->lang_code = $request->languagecode;
                $lang->group_langname = $translate->group_langname;
                $lang->key = $translate->key;
                $lang->value = $value;
                $lang->save();
            }
            if ($request->has('is_default') && $request->is_default == true) {

                $data['default_lang'] = removeSpaces($create->languagecode);
                $this->updateSettings($data);
            }

            $geolocation = GeoIP::getLocation(request()->getClientIp());
            $agent = new Agent();
            $activity = new EmployeeActivity();
            $activity->user_id = Auth::user()->id;
            $activity->activity_type = 'Created a language';
            $activity->ip_address = $geolocation->ip;
            $activity->browser = $agent->browser();
            $activity->device = $agent->device();
            $activity->save();
            return response()->json(['success' => 'Created Successfully', 'languagecode' => $request->languagecode]);
        }
    }

    /**	
     * Display the specified resource.	
     *	
     * @param  $lang Language code	
     * @param  $group Translate group	
     * @return \Illuminate\Http\Response	
     */
    public function translateData(Request $request, $code)
    {
        $language = Languages::where('languagecode', $code)->firstOrFail();

        $translates = Translate::where('lang_code', $language->languagecode)->get();

        $data['translates'] = $translates;
        $data['language'] = $language;
        return $data;
    }



    /**	
     * Display the specified resource.	
     *	
     * @param  $lang Language code	
     * @param  $group Translate group	
     * @return \Illuminate\Http\Response	
     */
    public function translate(Request $request, $code, $group = null)
    {
        // $this->authorize('Languages Translate');
        $language = Languages::where('languagecode', $code)->firstOrFail();
        if ($request->input('search')) {
            $q = $request->input('search');
            $groups = collect([
                (object) ["group_langname" => "Search results"],
            ]);
            $translates = Translate::where([['lang_code', $language->languagecode], ['key', 'like', '%' . $q . '%']])
                ->OrWhere([['lang_code', $language->languagecode], ['value', 'like', '%' . $q . '%']])
                ->OrWhere([['lang_code', $language->languagecode], ['group_langname', 'like', '%' . $q . '%']])
                ->orderbyDesc('id')
                ->get();
            $active = "Search results";

            $data['active'] = $active;
            $data['groups'] = $groups;
            $data['translates'] = $translates;
        } elseif ($request->input('filter')) {
            abort_if($request->input('filter') != 'missing', 404);
            $groups = collect([
                (object) ["group_langname" => "missing translations"],
            ]);
            $translates = Translate::where([['lang_code', $language->languagecode], ['value', null]])->orderby('group_langname')->get();
            $active = "missing translations";

            $data['active'] = $active;
            $data['groups'] = $groups;
            $data['translates'] = $translates;
        } else {
            $groups = Translate::where('lang_code', $language->languagecode)->select('group_langname')->groupBy('group_langname')->orderBy('id', 'ASC')->get();
            if ($group != null) {

                $group = str_replace('-', ' ', $group);
                $translates = Translate::where('lang_code', $language->languagecode)->where('group_langname', $group)->get();
                abort_if($translates->count() < 1, 404);
                $active = $group;

                $data['active'] = $active;
                $data['group'] = $group;
                $data['translates'] = $translates;
            } else {
                $translates = Translate::where('lang_code', $language->languagecode)->where('group_langname', 'general')->get();

                $active = "general";

                $data['active'] = $active;
                $data['translates'] = $translates;
            }
        }
        $translates_count = Translate::where('lang_code', $language->languagecode)->where('value', null)->count();
        $data['groups'] = $groups;
        $data['language'] = $language;
        $data['translates_count'] = $translates_count;
        return $data;
    }


    /**
     * Update the translate.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Language  $language
     * @return \Illuminate\Http\Response
     */
    public function translateUpdate(Request $request, $id)
    {
        $language = Languages::where('id', $id)->first();
        if ($language == null) {
            // toastr()->error(__('Something went wrong please try again'));
            return back();
        }
        foreach ($request->values as $id => $value) {
            $translation = Translate::where('id', $id)->first();
            if ($translation != null) {
                $translation->value = $value;
                $translation->save();
            }
        }
        $geolocation = GeoIP::getLocation(request()->getClientIp());
        $agent = new Agent();
        $activity = new EmployeeActivity();
        $activity->user_id = Auth::user()->id;
        $activity->activity_type = 'Added translations to a language.';
        $activity->ip_address = $geolocation->ip;
        $activity->browser = $agent->browser();
        $activity->device = $agent->device();
        $activity->save();

        return response()->json(['success' => 'Updated Successfully']);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  $id language ID
     * @return \Illuminate\Http\Response
     */
    public function setDefault($id)
    {
        $language = Languages::find($id);
        // $language = Languages::find(decrypt($id));

        if ($language != null) {
            if (setting('default_lang') == $language->languagecode) {
                return response()->json(['error' => 'Language already marked as default']);
            } else {
                $data['default_lang'] = $language->languagecode;
                $this->updateSettings($data);
                $geolocation = GeoIP::getLocation(request()->getClientIp());
                $agent = new Agent();
                $activity = new EmployeeActivity();
                $activity->user_id = Auth::user()->id;
                $activity->activity_type = 'Changed default language settings.';
                $activity->ip_address = $geolocation->ip;
                $activity->browser = $agent->browser();
                $activity->device = $agent->device();
                $activity->save();
                return response()->json(['success' => 'Default language has updated Successfully']);
            }
        } else {
            return response()->json(['error' => 'language not exists']);
        }
    }


    /**
     *  Settings Save/Update.
     *
     * @return \Illuminate\Http\Response
     */
    private function updateSettings($data)
    {

        foreach ($data as $key => $val) {
            $setting = Setting::where('key', $key);
            if ($setting->exists())
                $setting->first()->update(['value' => $val]);
        }
    }



    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Language  $language
     * @return \Illuminate\Http\Response
     */
    public function destroy($language)
    {
        $language = Languages::find($language);

        if ($language->languagecode == setting('default_lang')) {

            return response()->json(['error' => 'Default language cannot be deleted']);
        }
        $file_path = public_path() . "" . '/client/assets/i18n/' . $language->languagecode  . '.json';
        if (file_exists($file_path)) {
            unlink($file_path);
        }

        $language->delete();
        $geolocation = GeoIP::getLocation(request()->getClientIp());
        $agent = new Agent();
        $activity = new EmployeeActivity();
        $activity->user_id = Auth::user()->id;
        $activity->activity_type = 'Deleted a language.';
        $activity->ip_address = $geolocation->ip;
        $activity->browser = $agent->browser();
        $activity->device = $agent->device();
        $activity->save();

        return response()->json(['success' => 'Deleted Successfully']);
    }
    public function autoTranslate(Request $request)
    {
        try {
            $source_language = 'en';
            $target_language = $request->lang;
            $translations = Translate::where('lang_code', $target_language)->get();

            $translate = new GoogleTranslate();
            $translate->setSource($source_language);
            $translate->setTarget($target_language);
            foreach ($translations as $translation) {
                if (!$translation->value) {
                    $translation->value = $translate->translate($translation->key ?? ''); // Replace with your desired logic
                    $translation->save();
                }
            }

            return Response::json(['success' => lang('Translations updated successfully.')]);
        } catch (\Throwable $th) {
            return Response::json(['error' => $th->getMessage()]);
        }
    }
}
